<?php
require_once 'config/database.php';
require_once 'config/auth.php';
require_once 'includes/functions.php';
require_once 'config/currency.php';

$db = getDB();
$courseId = $_GET['id'] ?? 0;

if (!$courseId) {
    header('Location: index.php');
    exit;
}
if (isLoggedIn()) {
    $role = $_SESSION['role'];
    if ($role == 'admin') {
        header('Location: admin/dashboard.php');
    } elseif ($role == 'instructor') {
        header('Location: instructor/dashboard.php');
    } elseif ($role == 'student') {
        header('Location: student/dashboard.php');
    } else {
        header('Location: home.php');
    }
    exit;
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    
    if (login($username, $password)) {
        $role = $_SESSION['role'];
        if ($role == 'admin') {
            header('Location: admin/dashboard.php');
        } elseif ($role == 'instructor') {
            header('Location: instructor/dashboard.php');
        } elseif ($role == 'student') {
            header('Location: student/dashboard.php');
        } else {
            header('Location: home.php');
        }
        exit;
    } else {
        $error = 'Invalid username or password';
    }
}

$page_title = 'Course Details - Mutalex Academy';

// Get course details first before any HTML output
$stmt = $db->prepare("
    SELECT c.*, cat.name as category_name, cat.icon as category_icon, cat.color as category_color,
           u.username as instructor_name, up.first_name, up.last_name, up.bio as instructor_bio,
           COUNT(DISTINCT ce.id) as enrollment_count,
           AVG(cr.rating) as average_rating,
           COUNT(DISTINCT cr.id) as review_count
    FROM courses c
    LEFT JOIN course_categories cat ON c.category_id = cat.id
    LEFT JOIN users u ON c.instructor_id = u.id
    LEFT JOIN user_profiles up ON u.id = up.user_id
    LEFT JOIN course_enrollments ce ON c.id = ce.course_id AND ce.status IN ('enrolled', 'in_progress', 'completed')
    LEFT JOIN course_reviews cr ON c.id = cr.course_id AND cr.status = 'approved'
    WHERE c.id = ? AND c.status = 'published'
    GROUP BY c.id
");
$stmt->execute([$courseId]);
$course = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$course) {
    $_SESSION['error_message'] = 'Course not found.';
    header('Location: index.php');
    exit;
}

$page_title = htmlspecialchars($course['title']) . ' - Mutalex Academy';

// Get course lessons/curriculum
$stmt = $db->prepare("
    SELECT * FROM course_lessons
    WHERE course_id = ?
    ORDER BY sort_order
");
$stmt->execute([$courseId]);
$lessons = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get course reviews
$stmt = $db->prepare("
    SELECT cr.*, u.username, up.first_name, up.last_name
    FROM course_reviews cr
    JOIN users u ON cr.student_id = u.id
    LEFT JOIN user_profiles up ON u.id = up.user_id
    WHERE cr.course_id = ? AND cr.status = 'approved'
    ORDER BY cr.created_at DESC
    LIMIT 10
");
$stmt->execute([$courseId]);
$reviews = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Check if user is enrolled
$userEnrolled = false;
$userProgress = 0;
if (isLoggedIn()) {
    $userId = $_SESSION['user_id'];
    $stmt = $db->prepare("
        SELECT status, progress_percentage
        FROM course_enrollments
        WHERE course_id = ? AND student_id = ?
    ");
    $stmt->execute([$courseId, $userId]);
    $enrollment = $stmt->fetch(PDO::FETCH_ASSOC);

    if ($enrollment) {
        $userEnrolled = true;
        $userProgress = $enrollment['progress_percentage'];
    }
}

// Get related courses
$stmt = $db->prepare("
    SELECT c.*, AVG(cr.rating) as avg_rating, COUNT(cr.id) as review_count
    FROM courses c
    LEFT JOIN course_reviews cr ON c.id = cr.course_id AND cr.status = 'approved'
    WHERE c.category_id = ? AND c.id != ? AND c.status = 'published'
    GROUP BY c.id
    ORDER BY c.created_at DESC
    LIMIT 3
");
$stmt->execute([$course['category_id'], $courseId]);
$relatedCourses = $stmt->fetchAll(PDO::FETCH_ASSOC);

include 'includes/header.php';
?>

<style>
        /* Professional Modern Design System */
        :root {
            /* Primary Brand Colors */
            --primary-50: #f0f9ff;
            --primary-100: #e0f2fe;
            --primary-200: #bae6fd;
            --primary-300: #7dd3fc;
            --primary-400: #38bdf8;
            --primary-500: #0ea5e9;
            --primary-600: #0284c7;
            --primary-700: #0369a1;
            --primary-800: #075985;
            --primary-900: #0c4a6e;

            /* Neutral Colors */
            --neutral-50: #fafafa;
            --neutral-100: #f5f5f5;
            --neutral-200: #e5e5e5;
            --neutral-300: #d4d4d4;
            --neutral-400: #a3a3a3;
            --neutral-500: #737373;
            --neutral-600: #525252;
            --neutral-700: #404040;
            --neutral-800: #262626;
            --neutral-900: #171717;

            /* Semantic Colors */
            --success: #10b981;
            --warning: #f59e0b;
            --error: #ef4444;
            --info: #3b82f6;

            /* Spacing & Layout */
            --space-xs: 0.25rem;
            --space-sm: 0.5rem;
            --space-md: 1rem;
            --space-lg: 1.5rem;
            --space-xl: 2rem;
            --space-2xl: 3rem;
            --space-3xl: 4rem;

            /* Border Radius */
            --radius-sm: 0.375rem;
            --radius-md: 0.5rem;
            --radius-lg: 0.75rem;
            --radius-xl: 1rem;
            --radius-2xl: 1.5rem;

            /* Shadows */
            --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
            --shadow-md: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
            --shadow-lg: 0 10px 15px -3px rgb(0 0 0 / 0.1), 0 4px 6px -4px rgb(0 0 0 / 0.1);
            --shadow-xl: 0 20px 25px -5px rgb(0 0 0 / 0.1), 0 8px 10px -6px rgb(0 0 0 / 0.1);
            --shadow-2xl: 0 25px 50px -12px rgb(0 0 0 / 0.25);

            /* Typography */
            --font-sans: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
            --font-mono: 'JetBrains Mono', 'Fira Code', 'SF Mono', Monaco, 'Cascadia Code', monospace;

            /* Transitions */
            --transition-fast: 150ms cubic-bezier(0.4, 0, 0.2, 1);
            --transition-normal: 300ms cubic-bezier(0.4, 0, 0.2, 1);
            --transition-slow: 500ms cubic-bezier(0.4, 0, 0.2, 1);
        }

        /* Global Styles */
        * {
            box-sizing: border-box;
        }

        body {
            font-family: var(--font-sans);
            background: linear-gradient(135deg, var(--neutral-50) 0%, var(--neutral-100) 100%);
            color: var(--neutral-800);
            line-height: 1.6;
            margin: 0;
            padding: 0;
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
        }

        /* Typography Scale */
        .text-xs { font-size: 0.75rem; line-height: 1rem; }
        .text-sm { font-size: 0.875rem; line-height: 1.25rem; }
        .text-base { font-size: 1rem; line-height: 1.5rem; }
        .text-lg { font-size: 1.125rem; line-height: 1.75rem; }
        .text-xl { font-size: 1.25rem; line-height: 1.75rem; }
        .text-2xl { font-size: 1.5rem; line-height: 2rem; }
        .text-3xl { font-size: 1.875rem; line-height: 2.25rem; }
        .text-4xl { font-size: 2.25rem; line-height: 2.5rem; }
        .text-5xl { font-size: 3rem; line-height: 1; }
        .text-6xl { font-size: 3.75rem; line-height: 1; }

        .font-light { font-weight: 300; }
        .font-normal { font-weight: 400; }
        .font-medium { font-weight: 500; }
        .font-semibold { font-weight: 600; }
        .font-bold { font-weight: 700; }
        .font-black { font-weight: 900; }

        /* Layout Components */
        .container {
            max-width: 1280px;
            margin: 0 auto;
            padding: 0 var(--space-lg);
        }

        .grid {
            display: grid;
            gap: var(--space-lg);
        }

        .grid-cols-1 { grid-template-columns: repeat(1, minmax(0, 1fr)); }
        .grid-cols-2 { grid-template-columns: repeat(2, minmax(0, 1fr)); }
        .grid-cols-3 { grid-template-columns: repeat(3, minmax(0, 1fr)); }
        .grid-cols-4 { grid-template-columns: repeat(4, minmax(0, 1fr)); }

        /* Card Components */
        .card {
            background: white;
            border-radius: var(--radius-xl);
            box-shadow: var(--shadow-lg);
            border: 1px solid var(--neutral-200);
            transition: all var(--transition-normal);
        }

        .card:hover {
            box-shadow: var(--shadow-xl);
            transform: translateY(-2px);
        }

        .card-glass {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        /* Button Components */
        .btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: var(--space-sm);
            padding: var(--space-sm) var(--space-lg);
            border-radius: var(--radius-lg);
            font-weight: 600;
            font-size: 0.875rem;
            text-decoration: none;
            border: none;
            cursor: pointer;
            transition: all var(--transition-fast);
            position: relative;
            overflow: hidden;
        }

        .btn::before {
            content: '';
            position: absolute;
            top: 0;
            left: -100%;
            width: 100%;
            height: 100%;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.2), transparent);
            transition: left var(--transition-slow);
        }

        .btn:hover::before {
            left: 100%;
        }

        .btn-primary {
            background: linear-gradient(135deg, var(--primary-500), var(--primary-600));
            color: white;
            box-shadow: 0 4px 14px 0 rgba(14, 165, 233, 0.3);
        }

        .btn-primary:hover {
            background: linear-gradient(135deg, var(--primary-600), var(--primary-700));
            box-shadow: 0 6px 20px 0 rgba(14, 165, 233, 0.4);
            transform: translateY(-1px);
        }

        .btn-secondary {
            background: white;
            color: var(--neutral-700);
            border: 1px solid var(--neutral-300);
            box-shadow: var(--shadow-sm);
        }

        .btn-secondary:hover {
            background: var(--neutral-50);
            border-color: var(--neutral-400);
            box-shadow: var(--shadow-md);
        }

        .btn-ghost {
            background: transparent;
            color: var(--primary-600);
            border: 1px solid var(--primary-200);
        }

        .btn-ghost:hover {
            background: var(--primary-50);
            border-color: var(--primary-300);
        }

        /* Form Components */
        .form-group {
            margin-bottom: var(--space-lg);
        }

        .form-label {
            display: block;
            font-weight: 500;
            color: var(--neutral-700);
            margin-bottom: var(--space-sm);
        }

        .form-input {
            width: 100%;
            padding: var(--space-md);
            border: 1px solid var(--neutral-300);
            border-radius: var(--radius-lg);
            font-size: 1rem;
            transition: all var(--transition-fast);
            background: white;
        }

        .form-input:focus {
            outline: none;
            border-color: var(--primary-500);
            box-shadow: 0 0 0 3px rgba(14, 165, 233, 0.1);
        }

        /* Badge Components */
        .badge {
            display: inline-flex;
            align-items: center;
            gap: var(--space-xs);
            padding: var(--space-xs) var(--space-sm);
            border-radius: var(--radius-xl);
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
        }

        .badge-primary {
            background: var(--primary-100);
            color: var(--primary-700);
        }

        .badge-success {
            background: rgba(16, 185, 129, 0.1);
            color: var(--success);
        }

        .badge-warning {
            background: rgba(245, 158, 11, 0.1);
            color: var(--warning);
        }

        /* Progress Components */
        .progress {
            width: 100%;
            height: 8px;
            background: var(--neutral-200);
            border-radius: var(--radius-xl);
            overflow: hidden;
            position: relative;
        }

        .progress-bar {
            height: 100%;
            background: linear-gradient(90deg, var(--success), var(--primary-500));
            border-radius: var(--radius-xl);
            transition: width var(--transition-slow);
            position: relative;
        }

        .progress-bar::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.4), transparent);
            animation: progress-shimmer 2s infinite;
        }

        @keyframes progress-shimmer {
            0% { transform: translateX(-100%); }
            100% { transform: translateX(100%); }
        }

        /* Rating Components */
        .rating {
            display: flex;
            gap: var(--space-xs);
            align-items: center;
        }

        .rating .star {
            color: var(--neutral-300);
            transition: all var(--transition-fast);
        }

        .rating .star.filled {
            color: var(--warning);
            filter: drop-shadow(0 0 4px rgba(245, 158, 11, 0.3));
        }

        .rating .star:hover {
            transform: scale(1.1);
        }

        /* Tab Components */
        .tabs {
            background: white;
            border-radius: var(--radius-xl);
            box-shadow: var(--shadow-lg);
            overflow: hidden;
            margin-bottom: var(--space-2xl);
        }

        .tab-nav {
            display: flex;
            background: var(--neutral-50);
            border-bottom: 1px solid var(--neutral-200);
        }

        .tab-button {
            flex: 1;
            padding: var(--space-lg) var(--space-xl);
            text-align: center;
            font-weight: 600;
            color: var(--neutral-600);
            border: none;
            background: transparent;
            cursor: pointer;
            transition: all var(--transition-fast);
            position: relative;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: var(--space-sm);
        }

        .tab-button:hover {
            background: var(--neutral-100);
            color: var(--neutral-800);
        }

        .tab-button.active {
            color: var(--primary-600);
            background: white;
        }

        .tab-button.active::after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 50%;
            transform: translateX(-50%);
            width: 40px;
            height: 3px;
            background: linear-gradient(90deg, var(--primary-500), var(--primary-600));
            border-radius: 2px;
        }

        .tab-content {
            padding: var(--space-2xl);
            display: none;
        }

        .tab-content.active {
            display: block;
            animation: tab-fade-in 0.3s ease-out;
        }

        @keyframes tab-fade-in {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* Course Header */
        .course-header {
            background: linear-gradient(135deg, var(--primary-600) 0%, var(--primary-800) 100%);
            color: white;
            padding: var(--space-3xl) 0;
            position: relative;
            overflow: hidden;
        }

        .course-header::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><defs><pattern id="grain" width="100" height="100" patternUnits="userSpaceOnUse"><circle cx="25" cy="25" r="1" fill="white" opacity="0.03"/><circle cx="75" cy="75" r="1" fill="white" opacity="0.03"/><circle cx="50" cy="10" r="0.5" fill="white" opacity="0.02"/><circle cx="10" cy="50" r="0.5" fill="white" opacity="0.02"/><circle cx="90" cy="30" r="0.5" fill="white" opacity="0.02"/></pattern></defs><rect width="100" height="100" fill="url(%23grain)"/></svg>');
            pointer-events: none;
        }

        .course-title {
            font-size: clamp(2rem, 5vw, 3.5rem);
            font-weight: 800;
            line-height: 1.1;
            margin-bottom: var(--space-lg);
            background: linear-gradient(135deg, white 0%, rgba(255,255,255,0.9) 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .course-description {
            font-size: 1.25rem;
            line-height: 1.6;
            opacity: 0.9;
            max-width: 600px;
        }

        .course-meta {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: var(--space-lg);
            margin-top: var(--space-2xl);
        }

        .meta-item {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: var(--radius-xl);
            padding: var(--space-lg);
            display: flex;
            align-items: center;
            gap: var(--space-md);
            transition: all var(--transition-normal);
        }

        .meta-item:hover {
            background: rgba(255, 255, 255, 0.15);
            transform: translateY(-2px);
        }

        .meta-icon {
            width: 48px;
            height: 48px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            background: rgba(255, 255, 255, 0.2);
        }

        .meta-content h4 {
            font-size: 0.875rem;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.05em;
            opacity: 0.8;
            margin-bottom: var(--space-xs);
        }

        .meta-content p {
            font-size: 1rem;
            font-weight: 500;
            margin: 0;
        }

        /* Course Sidebar */
        .course-sidebar {
            background: white;
            border-radius: var(--radius-2xl);
            box-shadow: var(--shadow-2xl);
            padding: var(--space-2xl);
            position: sticky;
            top: var(--space-xl);
            border: 1px solid var(--neutral-200);
        }

        .course-price {
            text-align: center;
            margin-bottom: var(--space-xl);
            padding-bottom: var(--space-xl);
            border-bottom: 1px solid var(--neutral-200);
        }

        .price-amount {
            font-size: 2.5rem;
            font-weight: 800;
            color: var(--neutral-900);
            margin-bottom: var(--space-sm);
        }

        .price-label {
            font-size: 0.875rem;
            color: var(--neutral-600);
            font-weight: 500;
        }

        .enrollment-btn {
            width: 100%;
            padding: var(--space-lg);
            margin-bottom: var(--space-lg);
            font-size: 1.125rem;
            font-weight: 600;
        }

        .course-features {
            space-y: var(--space-md);
        }

        .feature-item {
            display: flex;
            align-items: center;
            gap: var(--space-md);
            padding: var(--space-sm) 0;
            color: var(--neutral-700);
        }

        .feature-icon {
            width: 20px;
            height: 20px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.75rem;
        }

        .feature-icon.success { background: rgba(16, 185, 129, 0.1); color: var(--success); }
        .feature-icon.primary { background: rgba(14, 165, 233, 0.1); color: var(--primary-600); }
        .feature-icon.warning { background: rgba(245, 158, 11, 0.1); color: var(--warning); }

        /* Content Sections */
        .content-section {
            background: white;
            border-radius: var(--radius-2xl);
            box-shadow: var(--shadow-lg);
            margin-bottom: var(--space-2xl);
            overflow: hidden;
            border: 1px solid var(--neutral-200);
        }

        .section-header {
            padding: var(--space-2xl);
            border-bottom: 1px solid var(--neutral-200);
            background: var(--neutral-50);
        }

        .section-title {
            font-size: 1.875rem;
            font-weight: 700;
            color: var(--neutral-900);
            margin: 0;
            display: flex;
            align-items: center;
            gap: var(--space-md);
        }

        .section-icon {
            width: 40px;
            height: 40px;
            border-radius: var(--radius-lg);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            background: var(--primary-100);
            color: var(--primary-600);
        }

        .section-content {
            padding: var(--space-2xl);
        }

        /* Curriculum */
        .lesson-item {
            background: var(--neutral-50);
            border-radius: var(--radius-xl);
            padding: var(--space-lg);
            margin-bottom: var(--space-md);
            border: 1px solid var(--neutral-200);
            transition: all var(--transition-normal);
            cursor: pointer;
        }

        .lesson-item:hover {
            background: white;
            box-shadow: var(--shadow-md);
            transform: translateX(4px);
        }

        .lesson-header {
            display: flex;
            align-items: flex-start;
            gap: var(--space-lg);
            margin-bottom: var(--space-md);
        }

        .lesson-icon {
            width: 56px;
            height: 56px;
            border-radius: var(--radius-xl);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            background: linear-gradient(135deg, var(--primary-500), var(--primary-600));
            color: white;
            flex-shrink: 0;
        }

        .lesson-content h4 {
            font-size: 1.125rem;
            font-weight: 600;
            color: var(--neutral-900);
            margin: 0 0 var(--space-sm) 0;
        }

        .lesson-meta {
            display: flex;
            gap: var(--space-lg);
            font-size: 0.875rem;
            color: var(--neutral-600);
        }

        /* Reviews */
        .review-item {
            background: var(--neutral-50);
            border-radius: var(--radius-xl);
            padding: var(--space-xl);
            margin-bottom: var(--space-lg);
            border: 1px solid var(--neutral-200);
        }

        .review-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: var(--space-lg);
        }

        .reviewer-info {
            display: flex;
            align-items: center;
            gap: var(--space-md);
        }

        .reviewer-avatar {
            width: 48px;
            height: 48px;
            border-radius: 50%;
            background: var(--primary-100);
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.25rem;
            color: var(--primary-600);
        }

        .reviewer-details h5 {
            font-size: 1rem;
            font-weight: 600;
            color: var(--neutral-900);
            margin: 0 0 var(--space-xs) 0;
        }

        .review-date {
            font-size: 0.875rem;
            color: var(--neutral-500);
        }

        /* Animations */
        .animate-fade-in {
            animation: fade-in 0.8s ease-out;
        }

        .animate-slide-up {
            animation: slide-up 0.8s ease-out;
        }

        .animate-scale-in {
            animation: scale-in 0.6s ease-out;
        }

        .animate-float {
            animation: float 6s ease-in-out infinite;
        }

        @keyframes fade-in {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        @keyframes slide-up {
            from { opacity: 0; transform: translateY(30px); }
            to { opacity: 1; transform: translateY(0); }
        }

        @keyframes scale-in {
            from { opacity: 0; transform: scale(0.95); }
            to { opacity: 1; transform: scale(1); }
        }

        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-10px); }
        }

        /* Enhanced Responsive Design */
        @media (max-width: 1200px) {
            .container { max-width: 100%; padding: 0 var(--space-lg); }
        }

        @media (max-width: 1024px) {
            .course-meta { grid-template-columns: repeat(2, 1fr); }
            .grid-cols-3 { grid-template-columns: repeat(2, 1fr); }
            .course-sidebar { margin-top: var(--space-3xl); }
        }

        @media (max-width: 768px) {
            .container { padding: 0 var(--space-md); }
            .course-header { padding: var(--space-2xl) 0; }
            .course-title { font-size: 2.5rem; }
            .course-meta { grid-template-columns: 1fr; gap: var(--space-md); }
            .grid-cols-2 { grid-template-columns: 1fr; }
            .grid-cols-3 { grid-template-columns: 1fr; }
            .tab-nav { flex-direction: column; gap: 0; }
            .course-sidebar { position: static; margin-top: var(--space-2xl); }
            .section-content { padding: var(--space-lg); }
            .section-header { padding: var(--space-lg); }
            .meta-item { padding: var(--space-md); }
            .price-amount { font-size: 2rem; }
        }

        @media (max-width: 640px) {
            .course-title { font-size: 2rem; line-height: 1.2; }
            .course-description { font-size: 1.125rem; }
            .btn { padding: var(--space-sm) var(--space-md); font-size: 0.875rem; }
            .section-title { font-size: 1.5rem; }
            .lesson-header { flex-direction: column; align-items: flex-start; gap: var(--space-md); }
            .lesson-icon { width: 48px; height: 48px; font-size: 1.25rem; }
            .review-header { flex-direction: column; align-items: flex-start; gap: var(--space-md); }
        }

        @media (max-width: 480px) {
            .course-title { font-size: 1.75rem; }
            .btn { padding: var(--space-sm); font-size: 0.8125rem; }
            .section-header { padding: var(--space-md); }
            .section-content { padding: var(--space-md); }
            .card { margin-bottom: var(--space-lg); }
            .badge { padding: var(--space-xs) var(--space-sm); font-size: 0.75rem; }
            .text-3xl { font-size: 1.5rem; }
            .text-4xl { font-size: 1.75rem; }
            .text-5xl { font-size: 2rem; }
            .text-6xl { font-size: 2.5rem; }
        }

        /* Touch-friendly interactions on mobile */
        @media (max-width: 768px) {
            .btn, .tab-button, .lesson-item {
                min-height: 44px;
            }

            .form-input {
                font-size: 16px; /* Prevents zoom on iOS */
            }
        }

        /* Accessibility */
        @media (prefers-reduced-motion: reduce) {
            *, *::before, *::after {
                animation-duration: 0.01ms !important;
                animation-iteration-count: 1 !important;
                transition-duration: 0.01ms !important;
            }
        }

        @media (prefers-contrast: high) {
            .card-glass {
                background: white !important;
                border: 2px solid var(--neutral-400) !important;
            }
        }

        /* Hover Effects */
        .hover-lift {
            transition: all var(--transition-normal);
        }

        .hover-lift:hover {
            transform: translateY(-4px);
            box-shadow: var(--shadow-xl);
        }

        .hover-glow:hover {
            box-shadow: 0 0 30px rgba(14, 165, 233, 0.3);
        }

        /* Gradient Text */
        .gradient-text {
            background: linear-gradient(135deg, var(--primary-600), var(--primary-800));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        /* Modern Scrollbar */
        ::-webkit-scrollbar {
            width: 8px;
        }

        ::-webkit-scrollbar-track {
            background: var(--neutral-100);
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb {
            background: var(--neutral-400);
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb:hover {
            background: var(--neutral-500);
        }

        /* Focus States */
        .btn:focus-visible,
        .form-input:focus-visible,
        .tab-button:focus-visible {
            outline: 2px solid var(--primary-500);
            outline-offset: 2px;
        }

        /* Print Styles */
        @media print {
            .course-sidebar,
            .tab-nav,
            .btn {
                display: none !important;
            }
        }
    </style>
</head>
<body class="antialiased">

<!-- Modern Hero Section -->
<section class="course-header">
    <div class="container">
        <div class="animate-fade-in">
            <!-- Category Badge -->
            <div class="flex justify-center mb-8">
                <div class="badge badge-primary animate-scale-in">
                    <i class="fas fa-tag"></i>
                    <?php echo htmlspecialchars($course['category_name'] ?? 'Software Development'); ?>
                </div>
            </div>

            <!-- Course Title -->
            <h1 class="course-title animate-slide-up"><?php echo htmlspecialchars($course['title']); ?></h1>

            <!-- Course Description -->
            <p class="course-description animate-slide-up" style="animation-delay: 0.2s;">
                <?php echo htmlspecialchars($course['short_description'] ?? substr($course['description'], 0, 150) . '...'); ?>
            </p>

            <!-- Course Meta Information -->
            <div class="course-meta animate-slide-up" style="animation-delay: 0.4s;">
                <div class="meta-item">
                    <div class="meta-icon">
                        <i class="fas fa-user-tie"></i>
                    </div>
                    <div class="meta-content">
                        <h4>Instructor</h4>
                        <p><?php echo htmlspecialchars($course['first_name'] && $course['last_name'] ? $course['first_name'] . ' ' . $course['last_name'] : $course['instructor_name']); ?></p>
                    </div>
                </div>

                <div class="meta-item">
                    <div class="meta-icon">
                        <i class="fas fa-star"></i>
                    </div>
                    <div class="meta-content">
                        <h4>Rating</h4>
                        <p><?php echo $course['average_rating'] ? number_format($course['average_rating'], 1) : '4.7'; ?> (<?php echo $course['review_count'] ?: '1,250'; ?> reviews)</p>
                    </div>
                </div>

                <div class="meta-item">
                    <div class="meta-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="meta-content">
                        <h4>Students</h4>
                        <p><?php echo number_format($course['enrollment_count']); ?> enrolled</p>
                    </div>
                </div>
            </div>

            <!-- CTA Section -->
            <div class="flex justify-center mt-12 animate-slide-up" style="animation-delay: 0.6s;">
                <div class="card card-glass p-8 max-w-md w-full text-center">
                    <div class="price-amount mb-6">
                        <?php echo $course['is_free'] ? 'Free' : formatCurrency($course['price'], $course['currency']); ?>
                    </div>
                    <div class="price-label mb-6">Lifetime Access</div>

                    <button id="enrollButton" class="enrollment-btn btn-primary">
                        <?php if (isLoggedIn()): ?>
                            <?php if ($userEnrolled): ?>
                                <a href="learn.php?id=<?php echo $course['id']; ?>" class="flex items-center justify-center gap-2">
                                    <i class="fas fa-play"></i>
                                    Continue Learning
                                </a>
                            <?php else: ?>
                                <?php if ($course['is_free']): ?>
                                    <span onclick="enrollCourse(<?php echo $course['id']; ?>)">Enroll Now - Free</span>
                                <?php else: ?>
                                    <span onclick="showPaymentModal(<?php echo $course['id']; ?>, '<?php echo addslashes($course['title']); ?>', <?php echo $course['price']; ?>, '<?php echo $course['currency'] ?: 'USD'; ?>')">Purchase Course</span>
                                <?php endif; ?>
                            <?php endif; ?>
                        <?php else: ?>
                            <a href="login.php?redirect=courses/detail.php?id=<?php echo $course['id']; ?>" class="flex items-center justify-center gap-2">
                                <i class="fas fa-sign-in-alt"></i>
                                Login to Enroll
                            </a>
                        <?php endif; ?>
                    </button>

                    <p class="text-sm text-neutral-300 mt-4 flex items-center justify-center gap-2">
                        <i class="fas fa-shield-alt"></i>
                        30-Day Money-Back Guarantee
                    </p>
                </div>
            </div>
        </div>
    </div>
</main>

    <!-- Modern Sidebar -->
    <aside class="course-sidebar animate-slide-up" style="animation-delay: 0.8s;">
        <!-- Instructor Card -->
        <div class="card card-glass mb-8">
            <div class="text-center mb-6">
                <div class="w-24 h-24 bg-primary-100 rounded-2xl overflow-hidden mx-auto mb-4">
                    <?php if (!empty($course['avatar'])): ?>
                        <img src="<?php echo htmlspecialchars($course['avatar']); ?>" alt="Instructor" class="w-full h-full object-cover">
                    <?php else: ?>
                        <div class="w-full h-full bg-primary-200 flex items-center justify-center">
                            <i class="fas fa-user text-3xl text-primary-600"></i>
                        </div>
                    <?php endif; ?>
                </div>
                <h3 class="text-xl font-bold text-neutral-900 mb-2"><?php echo htmlspecialchars($course['first_name'] && $course['last_name'] ? $course['first_name'] . ' ' . $course['last_name'] : $course['instructor_name']); ?></h3>
                <div class="badge badge-primary mb-4">Expert Instructor</div>
            </div>

            <?php if ($course['instructor_bio']): ?>
                <p class="text-neutral-600 text-sm leading-relaxed mb-6">
                    <?php echo htmlspecialchars($course['instructor_bio']); ?>
                </p>
            <?php endif; ?>

            <a href="#" class="btn btn-ghost w-full justify-center">
                <i class="fas fa-user-circle"></i>
                View Full Profile
            </a>
        </div>

        <!-- Course Features -->
        <div class="card card-glass">
            <div class="text-center mb-6">
                <div class="w-12 h-12 bg-success rounded-2xl flex items-center justify-center mx-auto mb-4">
                    <i class="fas fa-gift text-white text-xl"></i>
                </div>
                <h3 class="text-xl font-bold text-neutral-900">What You Get</h3>
            </div>

            <div class="space-y-4">
                <div class="feature-item">
                    <div class="feature-icon success">
                        <i class="fas fa-infinity"></i>
                    </div>
                    <div>
                        <h4 class="font-semibold text-neutral-900">Full Lifetime Access</h4>
                        <p class="text-sm text-neutral-600">Learn at your own pace</p>
                    </div>
                </div>

                <div class="feature-item">
                    <div class="feature-icon primary">
                        <i class="fas fa-certificate"></i>
                    </div>
                    <div>
                        <h4 class="font-semibold text-neutral-900">Complete Certification</h4>
                        <p class="text-sm text-neutral-600">Earn your certificate</p>
                    </div>
                </div>

                <div class="feature-item">
                    <div class="feature-icon warning">
                        <i class="fas fa-download"></i>
                    </div>
                    <div>
                        <h4 class="font-semibold text-neutral-900">Downloadable Assets</h4>
                        <p class="text-sm text-neutral-600">Source code & resources</p>
                    </div>
                </div>
            </div>
        </div>
    </aside>
</section>

<!-- Main Content Area -->
<main class="py-20 bg-neutral-50">
    <div class="container">
        <div class="grid grid-cols-1 lg:grid-cols-3 gap-12">

            <!-- Main Content Column -->
            <div class="lg:col-span-2 space-y-12">

                <!-- Course Statistics -->
                <div class="content-section animate-fade-in">
                    <div class="section-header">
                        <h2 class="section-title">
                            <div class="section-icon">
                                <i class="fas fa-chart-bar"></i>
                            </div>
                            Course Statistics
                        </h2>
                    </div>
                    <div class="grid grid-cols-2 lg:grid-cols-4 gap-6">
                        <div class="card card-glass p-6 text-center hover-lift">
                            <div class="w-12 h-12 bg-primary-100 rounded-2xl flex items-center justify-center mx-auto mb-4">
                                <i class="fas fa-users text-2xl text-primary-600"></i>
                            </div>
                            <div class="text-3xl font-black text-primary-600 mb-2 counter" data-target="<?php echo $course['enrollment_count']; ?>">0</div>
                            <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Students Enrolled</div>
                        </div>

                        <div class="card card-glass p-6 text-center hover-lift">
                            <div class="w-12 h-12 bg-success/10 rounded-2xl flex items-center justify-center mx-auto mb-4">
                                <i class="fas fa-clock text-2xl text-success"></i>
                            </div>
                            <div class="text-3xl font-black text-success mb-2 counter" data-target="<?php echo $course['duration_hours'] ?? 10; ?>">0</div>
                            <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Hours of Content</div>
                        </div>

                        <div class="card card-glass p-6 text-center hover-lift">
                            <div class="w-12 h-12 bg-warning/10 rounded-2xl flex items-center justify-center mx-auto mb-4">
                                <i class="fas fa-book text-2xl text-warning"></i>
                            </div>
                            <div class="text-3xl font-black text-warning mb-2 counter" data-target="<?php echo count($lessons); ?>">0</div>
                            <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Modules</div>
                        </div>

                        <div class="card card-glass p-6 text-center hover-lift">
                            <div class="w-12 h-12 bg-error/10 rounded-2xl flex items-center justify-center mx-auto mb-4">
                                <i class="fas fa-star text-2xl text-error"></i>
                            </div>
                            <div class="text-3xl font-black text-error mb-2 counter" data-target="<?php echo $course['review_count']; ?>">0</div>
                            <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Reviews</div>
                        </div>
                    </div>
                </div>

                <!-- Course Description -->
                <div class="content-section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <div class="section-icon">
                                <i class="fas fa-lightbulb"></i>
                            </div>
                            What You'll Learn
                        </h2>
                    </div>
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
                        <div class="card card-glass p-6">
                            <div class="flex items-start gap-4">
                                <div class="w-10 h-10 bg-success rounded-xl flex items-center justify-center flex-shrink-0">
                                    <i class="fas fa-check text-white text-sm font-bold"></i>
                                </div>
                                <span class="text-neutral-700 font-medium leading-relaxed">Deep understanding of Promises, Async/Await, and Generators for asynchronous tasks.</span>
                            </div>
                        </div>
                        <div class="card card-glass p-6">
                            <div class="flex items-start gap-4">
                                <div class="w-10 h-10 bg-success rounded-xl flex items-center justify-center flex-shrink-0">
                                    <i class="fas fa-check text-white text-sm font-bold"></i>
                                </div>
                                <span class="text-neutral-700 font-medium leading-relaxed">Mastering Array methods like `map`, `reduce`, and `filter` for cleaner code.</span>
                            </div>
                        </div>
                        <div class="card card-glass p-6">
                            <div class="flex items-start gap-4">
                                <div class="w-10 h-10 bg-success rounded-xl flex items-center justify-center flex-shrink-0">
                                    <i class="fas fa-check text-white text-sm font-bold"></i>
                                </div>
                                <span class="text-neutral-700 font-medium leading-relaxed">How to structure large-scale applications using modern module patterns.</span>
                            </div>
                        </div>
                        <div class="card card-glass p-6">
                            <div class="flex items-start gap-4">
                                <div class="w-10 h-10 bg-success rounded-xl flex items-center justify-center flex-shrink-0">
                                    <i class="fas fa-check text-white text-sm font-bold"></i>
                                </div>
                                <span class="text-neutral-700 font-medium leading-relaxed">Implementing error handling and debugging techniques in modern browsers.</span>
                            </div>
                        </div>
                        <div class="card card-glass p-6 md:col-span-2">
                            <div class="flex items-start gap-4">
                                <div class="w-10 h-10 bg-success rounded-xl flex items-center justify-center flex-shrink-0">
                                    <i class="fas fa-check text-white text-sm font-bold"></i>
                                </div>
                                <span class="text-neutral-700 font-medium leading-relaxed">Practical use of cutting-edge ES2024+ features in real-world projects.</span>
                            </div>
                        </div>
                    </div>

                    <div class="bg-neutral-50 rounded-2xl p-8 border border-neutral-200">
                        <h3 class="text-2xl font-bold text-neutral-900 mb-6">Course Overview</h3>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                            <div>
                                <p class="text-neutral-700 leading-relaxed mb-4">
                                    This course is designed for developers who already have a basic understanding of JavaScript and want to level up their skills to a professional standard. We start with fundamental concepts and quickly dive into advanced topics, ensuring you build both theoretical knowledge and practical experience.
                                </p>
                            </div>
                            <div>
                                <p class="text-neutral-700 leading-relaxed">
                                    Each module includes coding exercises, quizzes, and a final capstone project. You will build several mini-projects throughout the course, including a collaborative to-do list, a real-time chat application, and a complex data visualization dashboard.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Curriculum Section -->
                <div class="content-section">
                    <div class="section-header">
                        <h2 class="section-title">
                            <div class="section-icon">
                                <i class="fas fa-list-ul"></i>
                            </div>
                            Course Curriculum
                        </h2>
                    </div>
                    <div class="space-y-4">
                        <?php foreach ($lessons as $index => $lesson): ?>
                            <div class="lesson-item animate-slide-up" style="animation-delay: <?php echo $index * 0.1; ?>s;">
                                <div class="lesson-header">
                                    <div class="lesson-icon">
                                        <i class="fas fa-<?php
                                            echo $lesson['lesson_type'] === 'video' ? 'play-circle' :
                                                ($lesson['lesson_type'] === 'quiz' ? 'question-circle' :
                                                ($lesson['lesson_type'] === 'assignment' ? 'pencil-alt' : 'file-alt'));
                                        ?>"></i>
                                    </div>
                                    <div class="lesson-content">
                                        <div class="flex items-start justify-between mb-3">
                                            <h4><?php echo htmlspecialchars($lesson['title']); ?></h4>
                                            <?php if ($lesson['is_preview']): ?>
                                                <span class="badge badge-success">
                                                    <i class="fas fa-eye"></i>
                                                    Preview
                                                </span>
                                            <?php endif; ?>
                                        </div>
                                        <p class="text-neutral-600 mb-4"><?php echo htmlspecialchars($lesson['description'] ?? 'Lesson content description'); ?></p>
                                        <div class="lesson-meta">
                                            <span><i class="fas fa-tag"></i> <?php echo ucfirst($lesson['lesson_type'] ?? 'lesson'); ?></span>
                                            <?php if ($lesson['estimated_time']): ?>
                                                <span><i class="fas fa-clock"></i> <?php echo $lesson['estimated_time']; ?> min</span>
                                            <?php endif; ?>
                                            <span><i class="fas fa-play-circle"></i> Lesson <?php echo $index + 1; ?></span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>

                <!-- Course Tabs -->
                <div class="tabs animate-fade-in">
                    <nav class="tab-nav">
                        <button onclick="switchTab('overview')" id="overview-tab" class="tab-button active">
                            <i class="fas fa-info-circle"></i>
                            <span>Overview</span>
                        </button>
                        <button onclick="switchTab('instructor')" id="instructor-tab" class="tab-button">
                            <i class="fas fa-user-tie"></i>
                            <span>Instructor</span>
                        </button>
                        <button onclick="switchTab('reviews')" id="reviews-tab" class="tab-button">
                            <i class="fas fa-star"></i>
                            <span>Reviews</span>
                        </button>
                    </nav>

            <!-- Tab Content -->
            <div class="tab-content active" id="overview-content">
                <!-- Overview Tab -->
                <div class="section-content">
                    <div class="space-y-8">
                        <?php if ($course['learning_objectives']): ?>
                            <div class="content-section animate-slide-up">
                                <div class="section-header">
                                    <h3 class="section-title">
                                        <div class="section-icon">
                                            <i class="fas fa-lightbulb"></i>
                                        </div>
                                        What you'll learn
                                    </h3>
                                </div>
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <?php
                                    $objectives = json_decode($course['learning_objectives'], true);
                                    if ($objectives):
                                        foreach ($objectives as $index => $objective):
                                    ?>
                                        <div class="card card-glass p-6 animate-fade-in hover-lift" style="animation-delay: <?php echo $index * 0.1; ?>s;">
                                            <div class="flex items-start gap-4">
                                                <div class="w-10 h-10 bg-success rounded-xl flex items-center justify-center flex-shrink-0">
                                                    <i class="fas fa-check text-white text-sm font-bold"></i>
                                                </div>
                                                <span class="text-neutral-700 font-medium leading-relaxed"><?php echo htmlspecialchars($objective); ?></span>
                                            </div>
                                        </div>
                                    <?php endforeach; endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>

                        <?php if ($course['prerequisites']): ?>
                            <div class="content-section animate-slide-up">
                                <div class="section-header">
                                    <h3 class="section-title">
                                        <div class="section-icon" style="background: rgba(245, 158, 11, 0.1); color: var(--warning);">
                                            <i class="fas fa-exclamation-triangle"></i>
                                        </div>
                                        Prerequisites
                                    </h3>
                                </div>
                                <div class="bg-warning/5 border border-warning/20 rounded-xl p-6">
                                    <p class="text-neutral-700 leading-relaxed"><?php echo nl2br(htmlspecialchars($course['prerequisites'])); ?></p>
                                </div>
                            </div>
                        <?php endif; ?>

                        <?php if ($course['tags']): ?>
                            <div class="content-section animate-slide-up">
                                <div class="section-header">
                                    <h3 class="section-title">
                                        <div class="section-icon" style="background: rgba(168, 85, 247, 0.1); color: #a855f7;">
                                            <i class="fas fa-tags"></i>
                                        </div>
                                        Course Tags
                                    </h3>
                                </div>
                                <div class="flex flex-wrap gap-3">
                                    <?php
                                    $tags = json_decode($course['tags'], true);
                                    if ($tags):
                                        foreach ($tags as $index => $tag):
                                    ?>
                                        <span class="badge badge-primary animate-fade-in hover-lift cursor-pointer" style="animation-delay: <?php echo $index * 0.05; ?>s;">
                                            <i class="fas fa-hashtag"></i>
                                            <?php echo htmlspecialchars($tag); ?>
                                        </span>
                                    <?php endforeach; endif; ?>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Curriculum Tab -->
                <div class="tab-content" id="curriculum-content">
                    <div class="section-content">
                        <?php if (!empty($lessons)): ?>
                            <div class="space-y-6">
                                <?php foreach ($lessons as $index => $lesson): ?>
                                    <div class="lesson-item animate-slide-up" style="animation-delay: <?php echo $index * 0.1; ?>s;">
                                        <div class="lesson-header">
                                            <div class="lesson-icon">
                                                <i class="fas fa-<?php
                                                    echo $lesson['lesson_type'] === 'video' ? 'play-circle' :
                                                           ($lesson['lesson_type'] === 'quiz' ? 'question-circle' :
                                                           ($lesson['lesson_type'] === 'assignment' ? 'pencil-alt' : 'file-alt'));
                                                ?>"></i>
                                            </div>
                                            <div class="lesson-content">
                                                <div class="flex items-start justify-between mb-3">
                                                    <h4><?php echo htmlspecialchars($lesson['title']); ?></h4>
                                                    <?php if ($lesson['is_preview']): ?>
                                                        <span class="badge badge-success">
                                                            <i class="fas fa-eye"></i>
                                                            Preview
                                                        </span>
                                                    <?php endif; ?>
                                                </div>
                                                <p class="text-neutral-600 mb-4"><?php echo htmlspecialchars($lesson['description']); ?></p>
                                                <div class="lesson-meta">
                                                    <span><i class="fas fa-tag"></i> <?php echo ucfirst($lesson['lesson_type']); ?></span>
                                                    <?php if ($lesson['estimated_time']): ?>
                                                        <span><i class="fas fa-clock"></i> <?php echo $lesson['estimated_time']; ?> min</span>
                                                    <?php endif; ?>
                                                    <span><i class="fas fa-play-circle"></i> Lesson <?php echo $index + 1; ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>

                            <!-- Curriculum Summary -->
                            <div class="card card-glass mt-8 p-8">
                                <div class="grid grid-cols-1 md:grid-cols-3 gap-8 text-center">
                                    <div>
                                        <div class="text-4xl font-black text-primary mb-2"><?php echo count($lessons); ?></div>
                                        <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Total Lessons</div>
                                    </div>
                                    <div>
                                        <div class="text-4xl font-black text-success mb-2"><?php echo array_sum(array_column($lessons, 'estimated_time')); ?>m</div>
                                        <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Total Duration</div>
                                    </div>
                                    <div>
                                        <div class="text-4xl font-black text-warning mb-2"><?php echo count(array_filter($lessons, function($l) { return $l['is_preview']; })); ?></div>
                                        <div class="text-sm font-semibold text-neutral-600 uppercase tracking-wider">Preview Lessons</div>
                                    </div>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-20">
                                <div class="w-24 h-24 bg-neutral-200 rounded-2xl flex items-center justify-center mx-auto mb-8">
                                    <i class="fas fa-book-open text-4xl text-neutral-400"></i>
                                </div>
                                <h4 class="text-2xl font-bold text-neutral-900 mb-4">Curriculum Coming Soon</h4>
                                <p class="text-neutral-600 max-w-md mx-auto">The course content is being carefully prepared and will be available soon. Stay tuned!</p>
                                <div class="mt-8">
                                    <div class="inline-flex items-center gap-2 bg-neutral-100 px-6 py-3 rounded-xl">
                                        <i class="fas fa-clock text-warning"></i>
                                        <span class="text-sm font-medium text-neutral-700">Expected release: Soon</span>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Instructor Tab -->
                <div class="tab-content" id="instructor-content">
                    <div class="section-content">
                        <div class="content-section animate-fade-in">
                            <div class="section-header">
                                <h3 class="section-title">
                                    <div class="section-icon">
                                        <i class="fas fa-user-tie"></i>
                                    </div>
                                    Meet Your Instructor
                                </h3>
                            </div>

                            <div class="card p-8">
                                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                                    <div class="text-center">
                                        <div class="relative inline-block">
                                            <div class="w-32 h-32 bg-primary-100 rounded-2xl overflow-hidden mx-auto mb-4">
                                                <?php if (!empty($course['avatar'])): ?>
                                                    <img src="<?php echo htmlspecialchars($course['avatar']); ?>" alt="Instructor" class="w-full h-full object-cover">
                                                <?php else: ?>
                                                    <div class="w-full h-full bg-primary-200 flex items-center justify-center">
                                                        <i class="fas fa-user text-4xl text-primary-600"></i>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                            <div class="absolute -bottom-2 -right-2 w-8 h-8 bg-success rounded-xl flex items-center justify-center">
                                                <i class="fas fa-check text-white text-xs font-bold"></i>
                                            </div>
                                        </div>
                                        <div class="flex justify-center gap-2 mb-4">
                                            <span class="badge badge-primary">Expert Instructor</span>
                                            <span class="badge badge-success">Top Rated</span>
                                        </div>
                                    </div>

                                    <div class="lg:col-span-2">
                                        <h4 class="text-2xl font-bold text-neutral-900 mb-4"><?php echo htmlspecialchars($course['first_name'] && $course['last_name'] ? $course['first_name'] . ' ' . $course['last_name'] : $course['instructor_name']); ?></h4>

                                        <?php if ($course['instructor_bio']): ?>
                                            <div class="bg-neutral-50 rounded-xl p-6 mb-6 border border-neutral-200">
                                                <h5 class="text-lg font-semibold text-neutral-900 mb-3">About the Instructor</h5>
                                                <p class="text-neutral-700 leading-relaxed"><?php echo htmlspecialchars($course['instructor_bio']); ?></p>
                                            </div>
                                        <?php endif; ?>

                                        <!-- Instructor Stats -->
                                        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                                            <div class="card card-glass p-4 text-center hover-lift">
                                                <div class="text-2xl font-black text-primary mb-1"><?php echo $course['enrollment_count']; ?>+</div>
                                                <div class="text-xs font-semibold text-neutral-600 uppercase tracking-wider">Students Taught</div>
                                            </div>
                                            <div class="card card-glass p-4 text-center hover-lift">
                                                <div class="text-2xl font-black text-success mb-1"><?php echo number_format($course['average_rating'], 1); ?></div>
                                                <div class="text-xs font-semibold text-neutral-600 uppercase tracking-wider">Average Rating</div>
                                            </div>
                                            <div class="card card-glass p-4 text-center hover-lift">
                                                <div class="text-2xl font-black text-warning mb-1">5+</div>
                                                <div class="text-xs font-semibold text-neutral-600 uppercase tracking-wider">Years Experience</div>
                                            </div>
                                            <div class="card card-glass p-4 text-center hover-lift">
                                                <div class="text-2xl font-black text-error mb-1">15+</div>
                                                <div class="text-xs font-semibold text-neutral-600 uppercase tracking-wider">Courses Created</div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Reviews Tab -->
                <div class="tab-content" id="reviews-content">
                    <div class="section-content">
                        <?php if (!empty($reviews)): ?>
                            <!-- Overall Rating Card -->
                            <div class="content-section animate-fade-in mb-8">
                                <div class="section-header">
                                    <h3 class="section-title">
                                        <div class="section-icon" style="background: rgba(245, 158, 11, 0.1); color: var(--warning);">
                                            <i class="fas fa-star"></i>
                                        </div>
                                        Student Reviews
                                    </h3>
                                </div>

                                <div class="card p-8">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                                        <div class="text-center">
                                            <div class="text-6xl font-black text-neutral-900 mb-4"><?php echo number_format($course['average_rating'], 1); ?></div>
                                            <div class="rating mb-4">
                                                <?php for ($i = 1; $i <= 5; $i++): ?>
                                                    <i class="fas fa-star star <?php echo $i <= round($course['average_rating']) ? 'filled' : ''; ?>"></i>
                                                <?php endfor; ?>
                                            </div>
                                            <div class="text-lg font-bold text-neutral-700 mb-2"><?php echo $course['review_count']; ?> Reviews</div>
                                            <div class="text-sm text-neutral-600">Based on verified student feedback</div>
                                        </div>

                                        <div>
                                            <div class="space-y-3">
                                                <?php for ($i = 5; $i >= 1; $i--): ?>
                                                    <div class="flex items-center gap-4">
                                                        <div class="flex items-center gap-2 min-w-0">
                                                            <span class="text-sm font-medium text-neutral-700"><?php echo $i; ?></span>
                                                            <i class="fas fa-star text-warning text-sm"></i>
                                                        </div>
                                                        <div class="flex-1 bg-neutral-200 rounded-full h-2">
                                                            <div class="bg-warning h-2 rounded-full" style="width: <?php echo ($i <= round($course['average_rating'])) ? '80%' : '20%'; ?>"></div>
                                                        </div>
                                                        <span class="text-sm text-neutral-600 min-w-0"><?php echo rand(5, 25); ?>%</span>
                                                    </div>
                                                <?php endfor; ?>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Individual Reviews -->
                            <div class="space-y-6">
                                <?php foreach ($reviews as $index => $review): ?>
                                    <div class="review-item animate-slide-up" style="animation-delay: <?php echo $index * 0.1; ?>s;">
                                        <div class="review-header">
                                            <div class="reviewer-info">
                                                <div class="reviewer-avatar">
                                                    <i class="fas fa-user"></i>
                                                </div>
                                                <div class="reviewer-details">
                                                    <h5><?php echo htmlspecialchars($review['first_name'] && $review['last_name'] ? $review['first_name'] . ' ' . $review['last_name'] : $review['username']); ?></h5>
                                                    <div class="rating">
                                                        <?php for ($i = 1; $i <= 5; $i++): ?>
                                                            <i class="fas fa-star star <?php echo $i <= $review['rating'] ? 'filled' : ''; ?>"></i>
                                                        <?php endfor; ?>
                                                    </div>
                                                </div>
                                            </div>
                                            <div class="badge badge-neutral">
                                                <i class="fas fa-calendar-alt"></i>
                                                <?php echo date('M j, Y', strtotime($review['created_at'])); ?>
                                            </div>
                                        </div>

                                        <?php if ($review['review_title']): ?>
                                            <h4 class="text-xl font-bold text-neutral-900 mb-4 italic">"<?php echo htmlspecialchars($review['review_title']); ?>"</h4>
                                        <?php endif; ?>

                                        <?php if ($review['review_text']): ?>
                                            <div class="bg-neutral-50 rounded-xl p-6 border border-neutral-200">
                                                <p class="text-neutral-700 leading-relaxed"><?php echo htmlspecialchars($review['review_text']); ?></p>
                                            </div>
                                        <?php endif; ?>

                                        <!-- Review Actions -->
                                        <div class="flex items-center gap-4 mt-6 pt-6 border-t border-neutral-200">
                                            <button class="btn btn-ghost">
                                                <i class="fas fa-thumbs-up"></i>
                                                Helpful
                                            </button>
                                            <button class="btn btn-ghost">
                                                <i class="fas fa-reply"></i>
                                                Reply
                                            </button>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>

                            <!-- Load More Reviews -->
                            <?php if (count($reviews) >= 10): ?>
                                <div class="text-center mt-12">
                                    <button class="btn btn-primary">
                                        <i class="fas fa-plus"></i>
                                        <span>Load More Reviews</span>
                                    </button>
                                </div>
                            <?php endif; ?>
                        <?php else: ?>
                            <div class="text-center py-20">
                                <div class="w-24 h-24 bg-neutral-200 rounded-2xl flex items-center justify-center mx-auto mb-8">
                                    <i class="fas fa-comments text-4xl text-neutral-400"></i>
                                </div>
                                <h4 class="text-2xl font-bold text-neutral-900 mb-4">No Reviews Yet</h4>
                                <p class="text-neutral-600 max-w-md mx-auto mb-8">Be the first to share your experience after completing this course!</p>
                                <div class="inline-flex items-center gap-2 bg-neutral-100 px-6 py-3 rounded-xl">
                                    <i class="fas fa-star text-warning"></i>
                                    <span class="text-sm font-medium text-neutral-700">Reviews will appear here</span>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<!-- Related Courses -->
<?php if (!empty($relatedCourses)): ?>
<section class="py-20" style="background: linear-gradient(135deg, var(--neutral-50) 0%, var(--neutral-100) 100%);">
    <div class="container">
        <div class="text-center mb-16 animate-fade-in">
            <div class="inline-flex items-center gap-4 bg-primary-100 px-6 py-3 rounded-2xl mb-6">
                <div class="w-8 h-8 bg-primary-500 rounded-xl flex items-center justify-center">
                    <i class="fas fa-compass text-white"></i>
                </div>
                <span class="text-sm font-bold text-primary-700 uppercase tracking-wider">Explore More</span>
            </div>
            <h3 class="text-4xl font-black text-neutral-900 mb-4">Related Courses</h3>
            <p class="text-xl text-neutral-600 max-w-2xl mx-auto">Discover more courses in this category to expand your knowledge</p>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            <?php foreach ($relatedCourses as $index => $related): ?>
                <div class="card overflow-hidden hover-lift animate-slide-up group" style="animation-delay: <?php echo $index * 0.1; ?>s;">
                    <div class="relative h-56 overflow-hidden">
                        <?php if ($related['thumbnail']): ?>
                            <img src="<?php echo htmlspecialchars($related['thumbnail']); ?>" alt="<?php echo htmlspecialchars($related['title']); ?>" class="w-full h-full object-cover transition-transform duration-500 group-hover:scale-110">
                        <?php else: ?>
                            <div class="w-full h-full bg-primary-100 flex items-center justify-center relative">
                                <div class="absolute inset-0 bg-gradient-to-br from-primary-200/50 to-transparent"></div>
                                <i class="fas fa-graduation-cap text-5xl text-primary-600 relative z-10 animate-float"></i>
                            </div>
                        <?php endif; ?>
                        <div class="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                        <div class="absolute top-4 left-4">
                            <span class="badge" style="background: rgba(255,255,255,0.9); color: var(--neutral-800);">
                                <i class="fas fa-tag"></i>
                                Related
                            </span>
                        </div>
                    </div>

                    <div class="p-8">
                        <h4 class="text-xl font-bold text-neutral-900 mb-4 leading-tight group-hover:text-primary-600 transition-colors duration-300">
                            <a href="detail.php?id=<?php echo $related['id']; ?>" class="hover:text-primary-600 transition-colors duration-300"><?php echo htmlspecialchars($related['title']); ?></a>
                        </h4>

                        <?php if ($related['avg_rating']): ?>
                            <div class="flex items-center gap-3 mb-6">
                                <div class="rating">
                                    <?php for ($i = 1; $i <= 5; $i++): ?>
                                        <i class="fas fa-star star <?php echo $i <= round($related['avg_rating']) ? 'filled' : ''; ?>"></i>
                                    <?php endfor; ?>
                                </div>
                                <span class="text-sm font-bold text-neutral-700"><?php echo number_format($related['avg_rating'], 1); ?></span>
                                <span class="text-sm text-neutral-600">(<?php echo $related['review_count']; ?> reviews)</span>
                            </div>
                        <?php endif; ?>

                        <div class="flex items-center justify-between">
                            <div class="flex items-center gap-2">
                                <i class="fas fa-users text-primary-600"></i>
                                <span class="text-sm text-neutral-600">Students enrolled</span>
                            </div>
                            <a href="detail.php?id=<?php echo $related['id']; ?>" class="btn btn-ghost">
                                View Course
                                <i class="fas fa-arrow-right"></i>
                            </a>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- View All Related Courses -->
        <div class="text-center mt-16">
            <a href="catalog.php?category=<?php echo $course['category_id']; ?>" class="btn btn-primary">
                <i class="fas fa-th-large"></i>
                <span>View All <?php echo htmlspecialchars($course['category_name']); ?> Courses</span>
                <i class="fas fa-arrow-right"></i>
            </a>
        </div>
    </div>
</section>
<?php endif; ?>

<!-- Payment Modal -->
<div id="paymentModal" class="fixed inset-0 bg-black bg-opacity-50 hidden items-center justify-center z-50">
    <div class="bg-white rounded-xl shadow-2xl max-w-2xl w-full mx-4 max-h-[90vh] overflow-y-auto">
        <div class="p-6">
            <div class="flex justify-between items-center mb-6">
                <h2 class="text-2xl font-bold text-gray-900">Purchase Course</h2>
                <button onclick="closePaymentModal()" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times text-xl"></i>
                </button>
            </div>
            <div id="paymentContent">
                <!-- Payment form will be loaded here -->
            </div>
        </div>
    </div>
</div>


<script>
// Counter animation for stats
document.addEventListener('DOMContentLoaded', function() {
    const counters = document.querySelectorAll('[data-target]');
    counters.forEach(counter => {
        const target = parseInt(counter.getAttribute('data-target'));
        const increment = target / 100;
        let current = 0;

        const updateCounter = () => {
            current += increment;
            if (current < target) {
                counter.textContent = Math.floor(current);
                requestAnimationFrame(updateCounter);
            } else {
                counter.textContent = target;
            }
        };

        // Start animation when element comes into view
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    updateCounter();
                    observer.unobserve(entry.target);
                }
            });
        });
        observer.observe(counter);
    });
});

// Toast notification system
function showToast(message, type = 'success') {
    // Remove existing toasts
    const existingToasts = document.querySelectorAll('.toast-notification');
    existingToasts.forEach(toast => toast.remove());

    // Create toast element
    const toast = document.createElement('div');
    toast.className = `toast-notification fixed top-4 right-4 z-50 p-4 rounded-lg shadow-lg transform translate-x-full transition-all duration-300 ${
        type === 'success' ? 'bg-green-500 text-white' : 'bg-red-500 text-white'
    }`;
    toast.innerHTML = `
        <div class="flex items-center gap-3">
            <i class="fas ${type === 'success' ? 'fa-check-circle' : 'fa-exclamation-triangle'}"></i>
            <span>${message}</span>
        </div>
    `;

    // Add to page
    document.body.appendChild(toast);

    // Animate in
    setTimeout(() => {
        toast.classList.remove('translate-x-full');
    }, 100);

    // Auto remove after 4 seconds
    setTimeout(() => {
        toast.classList.add('translate-x-full');
        setTimeout(() => {
            toast.remove();
        }, 300);
    }, 4000);
}

// Tab switching functionality
function switchTab(tabName) {
    // Hide all tab contents
    const contents = document.querySelectorAll('.tab-content');
    contents.forEach(content => {
        content.classList.remove('active');
        content.classList.add('hidden');
    });

    // Remove active class from all tab buttons
    const buttons = document.querySelectorAll('.tab-button');
    buttons.forEach(button => button.classList.remove('active'));

    // Show selected tab content with animation
    const selectedContent = document.getElementById(tabName + '-content');
    if (selectedContent) {
        selectedContent.classList.remove('hidden');
        selectedContent.classList.add('active', 'animate-fade-in');
    }

    // Activate selected tab button
    const selectedTab = document.getElementById(tabName + '-tab');
    if (selectedTab) {
        selectedTab.classList.add('active');
    }

    // Update tab indicator position
    updateTabIndicator(tabName);
}

function updateTabIndicator(tabName) {
    const indicator = document.getElementById('tab-indicator');
    const activeTab = document.getElementById(tabName + '-tab');

    if (indicator && activeTab) {
        const tabRect = activeTab.getBoundingClientRect();
        const navRect = activeTab.parentElement.getBoundingClientRect();

        indicator.style.width = tabRect.width + 'px';
        indicator.style.left = (tabRect.left - navRect.left) + 'px';
    }
}

// Initialize tab indicator on page load
document.addEventListener('DOMContentLoaded', function() {
    updateTabIndicator('overview');
});

// Update tab indicator on window resize
window.addEventListener('resize', function() {
    const activeTab = document.querySelector('.tab-button.active');
    if (activeTab) {
        const tabName = activeTab.id.replace('-tab', '');
        updateTabIndicator(tabName);
    }
});

function enrollCourse(courseId) {
    // Show loading state
    const button = event.target.closest('button');
    const originalText = button.innerHTML;
    button.disabled = true;
    button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Enrolling...';

    fetch('api/enrollments.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({ course_id: courseId })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            // Show success message
            showToast('Successfully enrolled in the course!', 'success');
            // Reload page after a short delay to show the updated state
            setTimeout(() => {
                location.reload();
            }, 1500);
        } else {
            showToast('Error: ' + data.error, 'error');
            button.disabled = false;
            button.innerHTML = originalText;
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('An error occurred. Please try again.', 'error');
        button.disabled = false;
        button.innerHTML = originalText;
    });
}

function showPaymentModal(courseId, courseTitle, price, currency) {
    const modal = document.getElementById('paymentModal');
    const content = document.getElementById('paymentContent');

    content.innerHTML = `
        <div class="text-center mb-6">
            <h3 class="text-xl font-semibold text-gray-900 mb-2">${courseTitle}</h3>
            <p class="text-gray-600 mb-4">Complete your purchase to access this course</p>
            <div class="text-3xl font-bold text-blue-600">${currency} ${price.toFixed(2)}</div>
        </div>

        <form id="paymentForm" class="space-y-6">
            <input type="hidden" name="item_type" value="course">
            <input type="hidden" name="item_id" value="${courseId}">

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-2">Discount Code (Optional)</label>
                <input type="text" class="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent" name="discount_code" placeholder="Enter discount code">
                <div id="discountMessage" class="text-sm mt-1"></div>
            </div>

            <div>
                <label class="block text-sm font-medium text-gray-700 mb-3">Payment Method</label>
                <div class="space-y-3">
                    <div class="flex items-center">
                        <input class="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500" type="radio" name="gateway" value="paynow" id="paynow" checked>
                        <label class="ml-3 text-sm font-medium text-gray-700" for="paynow">
                            <i class="fas fa-mobile-alt mr-2 text-blue-500"></i>Paynow (Mobile Money)
                        </label>
                    </div>
                    <div class="flex items-center">
                        <input class="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500" type="radio" name="gateway" value="paypal" id="paypal">
                        <label class="ml-3 text-sm font-medium text-gray-700" for="paypal">
                            <i class="fab fa-paypal mr-2 text-blue-500"></i>PayPal
                        </label>
                    </div>
                </div>
            </div>

            <div id="finalAmount" class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <strong class="text-blue-900">Final Amount: ${currency} ${price.toFixed(2)}</strong>
            </div>

            <div class="text-center">
                <button type="submit" class="bg-blue-600 text-white px-8 py-3 rounded-lg font-semibold hover:bg-blue-700 transition duration-200 flex items-center justify-center gap-2 mx-auto" id="payButton">
                    <i class="fas fa-credit-card"></i>Proceed to Payment
                </button>
            </div>
        </form>
    `;

    // Handle discount code validation
    const discountInput = content.querySelector('input[name="discount_code"]');
    discountInput.addEventListener('blur', function() {
        validateDiscountCode(this.value, price, currency);
    });

    // Handle form submission
    const form = content.querySelector('#paymentForm');
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        processPayment(new FormData(this));
    });

    modal.classList.remove('hidden');
    modal.classList.add('flex');
}

function closePaymentModal() {
    const modal = document.getElementById('paymentModal');
    modal.classList.add('hidden');
    modal.classList.remove('flex');
}

function validateDiscountCode(code, originalPrice, currency) {
    if (!code.trim()) {
        document.getElementById('discountMessage').innerHTML = '';
        updateFinalAmount(originalPrice, 0, currency);
        return;
    }

    document.getElementById('discountMessage').innerHTML = '<small class="text-gray-500">Validating discount code...</small>';

    setTimeout(() => {
        if (code.toUpperCase().startsWith('TEST')) {
            const discountAmount = originalPrice * 0.1;
            document.getElementById('discountMessage').innerHTML = '<small class="text-green-600">10% discount applied!</small>';
            updateFinalAmount(originalPrice, discountAmount, currency);
        } else {
            document.getElementById('discountMessage').innerHTML = '<small class="text-red-600">Invalid discount code</small>';
            updateFinalAmount(originalPrice, 0, currency);
        }
    }, 500);
}

function updateFinalAmount(originalPrice, discountAmount, currency) {
    const finalAmount = originalPrice - discountAmount;
    document.getElementById('finalAmount').innerHTML = `<strong>Final Amount: ${currency} ${finalAmount.toFixed(2)}</strong>`;
}

function processPayment(formData) {
    const payButton = document.getElementById('payButton');
    const originalText = payButton.innerHTML;

    payButton.disabled = true;
    payButton.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Processing...';

    fetch('api/payments.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            if (data.redirect_url) {
                window.location.href = data.redirect_url;
            } else {
                showToast('Payment initiated successfully!', 'success');
                setTimeout(() => {
                    location.reload();
                }, 1500);
            }
        } else {
            showToast('Payment failed: ' + (data.error || 'Unknown error'), 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('An error occurred. Please try again.', 'error');
    })
    .finally(() => {
        payButton.disabled = false;
        payButton.innerHTML = originalText;
    });
}
</script>

<?php include 'includes/footer.php'; ?>